/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.security;

import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.lang.invoke.MethodHandles;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.Principal;
import java.security.PublicKey;
import java.security.SignatureException;
import java.time.Instant;
import java.util.Base64;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import javax.servlet.FilterChain;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.auth.BasicUserPrincipal;
import org.apache.http.protocol.HttpContext;
import org.apache.solr.client.solrj.SolrRequest;
import org.apache.solr.client.solrj.impl.Http2SolrClient;
import org.apache.solr.client.solrj.impl.HttpClientUtil;
import org.apache.solr.client.solrj.impl.HttpListenerFactory;
import org.apache.solr.client.solrj.impl.SolrHttpClientBuilder;
import org.apache.solr.client.solrj.request.GenericSolrRequest;
import org.apache.solr.client.solrj.response.SimpleSolrResponse;
import org.apache.solr.common.params.ModifiableSolrParams;
import org.apache.solr.common.params.SolrParams;
import org.apache.solr.common.util.ExecutorUtil;
import org.apache.solr.common.util.NamedList;
import org.apache.solr.common.util.StrUtils;
import org.apache.solr.common.util.SuppressForbidden;
import org.apache.solr.common.util.Utils;
import org.apache.solr.core.CoreContainer;
import org.apache.solr.request.SolrRequestInfo;
import org.apache.solr.security.AuthenticationPlugin;
import org.apache.solr.security.HttpClientBuilderPlugin;
import org.apache.solr.security.PublicKeyHandler;
import org.apache.solr.util.CryptoKeys;
import org.eclipse.jetty.client.api.Request;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PKIAuthenticationPlugin
extends AuthenticationPlugin
implements HttpClientBuilderPlugin {
    public static final String ACCEPT_VERSIONS = "solr.pki.acceptVersions";
    public static final String SEND_VERSION = "solr.pki.sendVersion";
    private static final Logger log = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private static final int MIN_TIMESTAMP_DIGITS = 10;
    private static final int MAX_TIMESTAMP_DIGITS = 13;
    private final Map<String, PublicKey> keyCache = new ConcurrentHashMap<String, PublicKey>();
    private final PublicKeyHandler publicKeyHandler;
    private final CoreContainer cores;
    private final LoadingCache<String, PKIHeaderData> validatedHeaderCache;
    private final LoadingCache<String, String> generatedV1TokenCache;
    private final LoadingCache<String, String> generatedV2TokenCache;
    private static final int MAX_VALIDITY = Integer.getInteger("pkiauth.ttl", 10000);
    private final String myNodeName;
    private final HttpHeaderClientInterceptor interceptor = new HttpHeaderClientInterceptor();
    private boolean interceptorRegistered = false;
    private boolean acceptPkiV1 = false;
    public static final String HEADER = "SolrAuth";
    public static final String HEADER_V2 = "SolrAuthV2";
    public static final String NODE_IS_USER = "$";
    public static final Principal CLUSTER_MEMBER_NODE = new BasicUserPrincipal("$");

    public static void withServerIdentity(boolean enabled) {
        SolrRequestInfo requestInfo = SolrRequestInfo.getRequestInfo();
        if (requestInfo != null) {
            requestInfo.setUseServerToken(enabled);
        }
        ExecutorUtil.setServerThreadFlag((Boolean)(enabled ? Boolean.valueOf(enabled) : null));
    }

    public boolean isInterceptorRegistered() {
        return this.interceptorRegistered;
    }

    public PKIAuthenticationPlugin(CoreContainer cores, String nodeName, PublicKeyHandler publicKeyHandler) {
        String[] versions;
        this.publicKeyHandler = publicKeyHandler;
        this.cores = cores;
        this.myNodeName = nodeName;
        this.validatedHeaderCache = Caffeine.newBuilder().maximumSize(1000L).expireAfterWrite((long)MAX_VALIDITY, TimeUnit.MILLISECONDS).build(this::decipherHeaderV2);
        long expireAfterTime = MAX_VALIDITY / 4;
        long shouldRefreshTime = Math.max(1L, expireAfterTime / 2L);
        this.generatedV1TokenCache = Caffeine.newBuilder().maximumSize(100L).refreshAfterWrite(shouldRefreshTime, TimeUnit.MILLISECONDS).expireAfterWrite(expireAfterTime, TimeUnit.MILLISECONDS).build(this::generateToken);
        this.generatedV2TokenCache = Caffeine.newBuilder().maximumSize(100L).refreshAfterWrite(shouldRefreshTime, TimeUnit.MILLISECONDS).expireAfterWrite(expireAfterTime, TimeUnit.MILLISECONDS).build(this::generateTokenV2);
        Set<String> knownPkiVersions = Set.of("v1", "v2");
        for (String version : versions = System.getProperty(ACCEPT_VERSIONS, "v2").split(",")) {
            if (!knownPkiVersions.contains(version)) {
                log.warn("Unknown protocol version [{}] specified in {}", (Object)version, (Object)ACCEPT_VERSIONS);
            }
            if (!"v1".equals(version)) continue;
            log.warn("System setting {} includes the deprecated v1, which should only be used for compatibility during rolling upgrades. After all servers have been upgraded, consider disabling this compatability layer.", (Object)ACCEPT_VERSIONS);
            this.acceptPkiV1 = true;
        }
    }

    @Override
    public void init(Map<String, Object> pluginConfig) {
    }

    @Override
    @SuppressForbidden(reason="Needs currentTimeMillis to compare against time in header")
    public boolean doAuthenticate(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws Exception {
        long receivedTime = System.currentTimeMillis();
        PKIHeaderData headerData = null;
        String headerV2 = request.getHeader(HEADER_V2);
        String headerV1 = request.getHeader(HEADER);
        if (headerV1 == null && headerV2 == null) {
            return this.sendError(response, true, "No PKI auth header was provided");
        }
        if (headerV2 != null) {
            int nodeNameEnd = headerV2.indexOf(32);
            if (nodeNameEnd <= 0) {
                return this.sendError(response, true, "Could not parse node name from SolrAuthV2 header.");
            }
            headerData = (PKIHeaderData)this.validatedHeaderCache.get((Object)headerV2);
        } else if (headerV1 != null && this.acceptPkiV1) {
            List authInfo = StrUtils.splitWS((String)headerV1, (boolean)false);
            if (authInfo.size() != 2) {
                return this.sendError(response, false, "Invalid SolrAuth header: " + headerV1);
            }
            headerData = this.decipherHeader((String)authInfo.get(0), (String)authInfo.get(1));
        }
        if (headerData == null) {
            return this.sendError(response, true, "Could not validate PKI header.");
        }
        long elapsed = receivedTime - headerData.timestamp;
        if (elapsed > (long)MAX_VALIDITY) {
            return this.sendError(response, true, "Expired key request timestamp, elapsed=" + elapsed);
        }
        Principal principal = NODE_IS_USER.equals(headerData.userName) ? CLUSTER_MEMBER_NODE : new BasicUserPrincipal(headerData.userName);
        this.numAuthenticated.inc();
        filterChain.doFilter((ServletRequest)this.wrapWithPrincipal(request, principal), (ServletResponse)response);
        return true;
    }

    private boolean sendError(HttpServletResponse response, boolean v2, String message) throws IOException {
        this.numErrors.mark();
        log.error(message);
        response.setHeader("WWW-Authenticate", v2 ? HEADER_V2 : HEADER);
        response.sendError(401, message);
        return false;
    }

    private PublicKey getOrFetchPublicKey(String nodeName) {
        PublicKey key = this.keyCache.get(nodeName);
        if (key == null) {
            log.debug("No key available for node: {} fetching now ", (Object)nodeName);
            key = this.fetchPublicKeyFromRemote(nodeName);
            log.debug("public key obtained {} ", (Object)key);
        }
        return key;
    }

    private PKIHeaderData decipherHeaderV2(String header) {
        byte[] sig;
        String nodeName = header.substring(0, header.indexOf(32));
        PublicKey key = this.getOrFetchPublicKey(nodeName);
        int sigStart = header.lastIndexOf(32);
        String data = header.substring(0, sigStart);
        PKIHeaderData rv = this.validateSignature(data, sig = Base64.getDecoder().decode(header.substring(sigStart + 1)), key, false);
        if (rv == null) {
            log.warn("Failed to verify signature, trying after refreshing the key ");
            key = this.fetchPublicKeyFromRemote(nodeName);
            rv = this.validateSignature(data, sig, key, true);
        }
        return rv;
    }

    private PKIHeaderData validateSignature(String data, byte[] sig, PublicKey key, boolean isRetry) {
        if (key == null) {
            log.warn("Key is null when attempting to validate signature; skipping...");
            return null;
        }
        try {
            if (CryptoKeys.verifySha256(data.getBytes(StandardCharsets.UTF_8), sig, key)) {
                int timestampStart = data.lastIndexOf(32);
                PKIHeaderData rv = new PKIHeaderData();
                String ts = data.substring(timestampStart + 1);
                try {
                    rv.timestamp = Long.parseLong(ts);
                }
                catch (NumberFormatException e) {
                    log.error("SolrAuthV2 header error, cannot parse {} as timestamp", (Object)ts);
                    return null;
                }
                rv.userName = data.substring(data.indexOf(32) + 1, timestampStart);
                return rv;
            }
            log.warn("Signature verification failed, signature or checksum does not match");
            return null;
        }
        catch (InvalidKeyException | SignatureException e) {
            String excMessage = e.getMessage();
            if (isRetry) {
                log.error("Signature validation on retry failed, likely key error: {}", (Object)excMessage);
            } else {
                log.info("Signature validation failed first attempt, likely key error: {}", (Object)excMessage);
            }
            return null;
        }
    }

    private PKIHeaderData decipherHeader(String nodeName, String cipherBase64) {
        PublicKey key = this.getOrFetchPublicKey(nodeName);
        PKIHeaderData header = PKIAuthenticationPlugin.parseCipher(cipherBase64, key, false);
        if (header == null) {
            log.warn("Failed to decrypt header, trying after refreshing the key ");
            key = this.fetchPublicKeyFromRemote(nodeName);
            return PKIAuthenticationPlugin.parseCipher(cipherBase64, key, true);
        }
        return header;
    }

    @VisibleForTesting
    static PKIHeaderData parseCipher(String cipher, PublicKey key, boolean isRetry) {
        byte[] bytes;
        try {
            bytes = CryptoKeys.decryptRSA(Base64.getDecoder().decode(cipher), key);
        }
        catch (Exception e) {
            if (isRetry) {
                log.error("Decryption failed on retry, key must be wrong", (Throwable)e);
            } else {
                log.info("Decryption failed on first attempt, will retry", (Throwable)e);
            }
            return null;
        }
        String s = new String(bytes, StandardCharsets.UTF_8).trim();
        int splitPoint = s.lastIndexOf(32);
        int timestampDigits = s.length() - 1 - splitPoint;
        if (splitPoint == -1 || timestampDigits < 10 || timestampDigits > 13) {
            log.warn("Invalid cipher {} deciphered data {}", (Object)cipher, (Object)s);
            return null;
        }
        PKIHeaderData headerData = new PKIHeaderData();
        try {
            headerData.timestamp = Long.parseLong(s.substring(splitPoint + 1));
            headerData.userName = s.substring(0, splitPoint);
            log.debug("Successfully decrypted header {} {}", (Object)headerData.userName, (Object)headerData.timestamp);
            return headerData;
        }
        catch (NumberFormatException e) {
            log.warn("Invalid cipher {}", (Object)cipher);
            return null;
        }
    }

    private boolean isInLiveNodes(String nodeName) {
        return this.cores.getZkController().getZkStateReader().getClusterState().getLiveNodes().contains(nodeName);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    PublicKey fetchPublicKeyFromRemote(String nodename) {
        if (!this.isInLiveNodes(nodename)) {
            log.warn("Unable to fetch public key for {} as it does not appear to be a Solr 'live node'", (Object)nodename);
            return null;
        }
        String url = this.cores.getZkController().getZkStateReader().getBaseUrlForNodeName(nodename);
        HttpEntity entity = null;
        try {
            ModifiableSolrParams solrParams = new ModifiableSolrParams();
            solrParams.add("wt", new String[]{"json"});
            solrParams.add("omitHeader", new String[]{"true"});
            GenericSolrRequest request = new GenericSolrRequest(SolrRequest.METHOD.GET, "/admin/info/key", (SolrParams)solrParams);
            log.debug("Fetching fresh public key from: {}", (Object)url);
            Http2SolrClient solrClient = this.cores.getDefaultHttpSolrClient();
            NamedList resp = ((SimpleSolrResponse)solrClient.requestWithBaseUrl(url, arg_0 -> ((GenericSolrRequest)request).process(arg_0))).getResponse();
            String key = (String)resp.get("key");
            if (key == null) {
                log.error("No key available from {}{}", (Object)url, (Object)"/admin/info/key");
                PublicKey publicKey = null;
                return publicKey;
            }
            log.info("New key obtained from node={}, key={}", (Object)nodename, (Object)key);
            PublicKey pubKey = CryptoKeys.deserializeX509PublicKey(key);
            this.keyCache.put(nodename, pubKey);
            PublicKey publicKey = pubKey;
            return publicKey;
        }
        catch (Exception e) {
            log.error("Exception trying to get public key from: {}", (Object)url, (Object)e);
            PublicKey publicKey = null;
            return publicKey;
        }
        finally {
            Utils.consumeFully(entity);
        }
    }

    @Override
    public void setup(Http2SolrClient client) {
        HttpListenerFactory.RequestResponseListener listener = new HttpListenerFactory.RequestResponseListener(){
            private static final String CACHED_REQUEST_USER_KEY = "cachedRequestUser";

            public void onQueued(Request request) {
                log.trace("onQueued: {}", (Object)request);
                if (PKIAuthenticationPlugin.this.cores.getAuthenticationPlugin() == null) {
                    log.trace("no authentication plugin, skipping");
                    return;
                }
                if (!PKIAuthenticationPlugin.this.cores.getAuthenticationPlugin().interceptInternodeRequest(request)) {
                    if (log.isDebugEnabled()) {
                        log.debug("{} secures this internode request", (Object)((Object)((Object)this)).getClass().getSimpleName());
                    }
                    this.cachePreFetchedUserOnJettyRequest(request);
                } else if (log.isDebugEnabled()) {
                    log.debug("{} secures this internode request", (Object)PKIAuthenticationPlugin.this.cores.getAuthenticationPlugin().getClass().getSimpleName());
                }
            }

            public void onBegin(Request request) {
                log.trace("onBegin: {}", (Object)request);
                Optional<String> preFetchedUser = this.getUserFromJettyRequest(request);
                if ("v1".equals(System.getProperty(PKIAuthenticationPlugin.SEND_VERSION))) {
                    preFetchedUser.map(arg_0 -> PKIAuthenticationPlugin.this.generatedV1TokenCache.get(arg_0)).ifPresent(token -> request.header(PKIAuthenticationPlugin.HEADER, token));
                } else {
                    preFetchedUser.map(arg_0 -> PKIAuthenticationPlugin.this.generatedV2TokenCache.get(arg_0)).ifPresent(token -> request.header(PKIAuthenticationPlugin.HEADER_V2, token));
                }
            }

            private void cachePreFetchedUserOnJettyRequest(Request request) {
                PKIAuthenticationPlugin.this.getUser().ifPresent(user -> request.attribute(CACHED_REQUEST_USER_KEY, user));
            }

            private Optional<String> getUserFromJettyRequest(Request request) {
                return Optional.ofNullable((String)request.getAttributes().get(CACHED_REQUEST_USER_KEY));
            }
        };
        client.addListenerFactory(() -> listener);
    }

    @Override
    public SolrHttpClientBuilder getHttpClientBuilder(SolrHttpClientBuilder builder) {
        HttpClientUtil.addRequestInterceptor((HttpRequestInterceptor)this.interceptor);
        this.interceptorRegistered = true;
        return builder;
    }

    public boolean needsAuthorization(HttpServletRequest req) {
        return req.getUserPrincipal() != CLUSTER_MEMBER_NODE;
    }

    private Optional<String> getUser() {
        SolrRequestInfo reqInfo = this.getRequestInfo();
        if (reqInfo != null && !reqInfo.useServerToken()) {
            Principal principal = reqInfo.getUserPrincipal();
            if (principal == null) {
                log.debug("generateToken: principal is null");
                return Optional.empty();
            }
            assert (principal.getName() != null);
            return Optional.of(principal.getName());
        }
        if (!this.isSolrThread()) {
            log.debug("generateToken: not a solr (server) thread");
            return Optional.empty();
        }
        return Optional.of(NODE_IS_USER);
    }

    @SuppressForbidden(reason="Needs currentTimeMillis to set current time in header")
    private String generateToken(String usr) {
        assert (usr != null);
        String s = usr + " " + System.currentTimeMillis();
        byte[] payload = s.getBytes(StandardCharsets.UTF_8);
        byte[] payloadCipher = this.publicKeyHandler.getKeyPair().encrypt(ByteBuffer.wrap(payload));
        String base64Cipher = Base64.getEncoder().encodeToString(payloadCipher);
        log.trace("generateToken: usr={} token={}", (Object)usr, (Object)base64Cipher);
        return this.myNodeName + " " + base64Cipher;
    }

    private String generateTokenV2(String user) {
        assert (user != null);
        String s = this.myNodeName + " " + user + " " + Instant.now().toEpochMilli();
        byte[] payload = s.getBytes(StandardCharsets.UTF_8);
        byte[] signature = this.publicKeyHandler.getKeyPair().signSha256(payload);
        String base64Signature = Base64.getEncoder().encodeToString(signature);
        return s + " " + base64Signature;
    }

    void setHeader(HttpRequest httpRequest) {
        if ("v1".equals(System.getProperty(SEND_VERSION))) {
            this.getUser().map(arg_0 -> this.generatedV1TokenCache.get(arg_0)).ifPresent(token -> httpRequest.setHeader(HEADER, token));
        } else {
            this.getUser().map(arg_0 -> this.generatedV2TokenCache.get(arg_0)).ifPresent(token -> httpRequest.setHeader(HEADER_V2, token));
        }
    }

    boolean isSolrThread() {
        return ExecutorUtil.isSolrServerThread();
    }

    SolrRequestInfo getRequestInfo() {
        return SolrRequestInfo.getRequestInfo();
    }

    @Override
    public void close() throws IOException {
        HttpClientUtil.removeRequestInterceptor((HttpRequestInterceptor)this.interceptor);
        this.interceptorRegistered = false;
    }

    @VisibleForTesting
    public String getPublicKey() {
        return this.publicKeyHandler.getKeyPair().getPublicKeyStr();
    }

    private class HttpHeaderClientInterceptor
    implements HttpRequestInterceptor {
        public void process(HttpRequest httpRequest, HttpContext httpContext) throws HttpException, IOException {
            if (PKIAuthenticationPlugin.this.cores.getAuthenticationPlugin() == null) {
                return;
            }
            if (!PKIAuthenticationPlugin.this.cores.getAuthenticationPlugin().interceptInternodeRequest(httpRequest, httpContext)) {
                if (log.isDebugEnabled()) {
                    log.debug("{} secures this internode request", (Object)this.getClass().getSimpleName());
                }
                PKIAuthenticationPlugin.this.setHeader(httpRequest);
            } else if (log.isDebugEnabled()) {
                log.debug("{} secures this internode request", (Object)PKIAuthenticationPlugin.this.cores.getAuthenticationPlugin().getClass().getSimpleName());
            }
        }
    }

    public static class PKIHeaderData {
        String userName;
        long timestamp;

        public String toString() {
            return "PKIHeaderData{userName='" + this.userName + "', timestamp=" + this.timestamp + "}";
        }
    }
}

