-- |
-- Module      : SAD.Parser.Lexer
-- Copyright   : (c) 2024, Marcel Schütz
-- License     : GPL-3
--
-- Generic lexing setup

{-# LANGUAGE MultiParamTypeClasses #-}
{-# LANGUAGE OverloadedStrings #-}

module SAD.Parser.Lexer where

import FTLex.Base
import FTLex.Position qualified as Pos
import FTLex.Message qualified as Msg
import Data.Text (Text)
import Data.Text qualified as Text
import Data.Set qualified as Set

import SAD.Core.Message qualified as Message

import Isabelle.Position qualified as Position
import Isabelle.Bytes qualified as Bytes
import Isabelle.Symbol qualified as Symbol
import Isabelle.Library qualified as Library
import FTLex.Position (noPos)


-- * PIDE Setup

instance Pos.Pos Position.T where
  noPos :: Position.T
  noPos = Position.none

  getNextPos :: Text -> Position.T -> Position.T
  getNextPos = Position.symbol_explode

  getPosOf :: Text -> Position.T -> Position.T
  getPosOf text pos =
    let newPos = Position.symbol_explode text pos
    in Position.range_position (pos, newPos)

  showPos :: Position.T -> Text
  showPos = Library.make_text . Position.here


instance Msg.Msg Position.T IO where
  errorLexer :: Position.T -> Text -> IO a
  errorLexer = Message.errorLexer


-- * Misc

-- | Supported Unicode code blocks
codeBlocks :: Set.Set UnicodeBlock
codeBlocks = Set.fromList [
    BasicLatin,
    Latin1Supplement,
    LatinExtendedA,
    LatinExtendedB
  ]

-- | Decode a "PIDE-encoded" byte string.
pideDecode :: Bytes.Bytes -> IO Text
pideDecode bytes = do
  let string = map Library.make_string . Symbol.explode $ bytes
  decodedString <- mapM decode string
  return $ Text.pack decodedString
  where
    decode "\\<zero>" = pure '\x01d7ec'
    decode "\\<one>" = pure '\x01d7ed'
    decode "\\<two>" = pure '\x01d7ee'
    decode "\\<three>" = pure '\x01d7ef'
    decode "\\<four>" = pure '\x01d7f0'
    decode "\\<five>" = pure '\x01d7f1'
    decode "\\<six>" = pure '\x01d7f2'
    decode "\\<seven>" = pure '\x01d7f3'
    decode "\\<eight>" = pure '\x01d7f4'
    decode "\\<nine>" = pure '\x01d7f5'
    decode "\\<A>" = pure '\x01d49c'
    decode "\\<B>" = pure '\x00212c'
    decode "\\<C>" = pure '\x01d49e'
    decode "\\<D>" = pure '\x01d49f'
    decode "\\<E>" = pure '\x002130'
    decode "\\<F>" = pure '\x002131'
    decode "\\<G>" = pure '\x01d4a2'
    decode "\\<H>" = pure '\x00210b'
    decode "\\<I>" = pure '\x002110'
    decode "\\<J>" = pure '\x01d4a5'
    decode "\\<K>" = pure '\x01d4a6'
    decode "\\<L>" = pure '\x002112'
    decode "\\<M>" = pure '\x002133'
    decode "\\<N>" = pure '\x01d4a9'
    decode "\\<O>" = pure '\x01d4aa'
    decode "\\<P>" = pure '\x01d4ab'
    decode "\\<Q>" = pure '\x01d4ac'
    decode "\\<R>" = pure '\x00211b'
    decode "\\<S>" = pure '\x01d4ae'
    decode "\\<T>" = pure '\x01d4af'
    decode "\\<U>" = pure '\x01d4b0'
    decode "\\<V>" = pure '\x01d4b1'
    decode "\\<W>" = pure '\x01d4b2'
    decode "\\<X>" = pure '\x01d4b3'
    decode "\\<Y>" = pure '\x01d4b4'
    decode "\\<Z>" = pure '\x01d4b5'
    decode "\\<a>" = pure '\x01d5ba'
    decode "\\<b>" = pure '\x01d5bb'
    decode "\\<c>" = pure '\x01d5bc'
    decode "\\<d>" = pure '\x01d5bd'
    decode "\\<e>" = pure '\x01d5be'
    decode "\\<f>" = pure '\x01d5bf'
    decode "\\<g>" = pure '\x01d5c0'
    decode "\\<h>" = pure '\x01d5c1'
    decode "\\<i>" = pure '\x01d5c2'
    decode "\\<j>" = pure '\x01d5c3'
    decode "\\<k>" = pure '\x01d5c4'
    decode "\\<l>" = pure '\x01d5c5'
    decode "\\<m>" = pure '\x01d5c6'
    decode "\\<n>" = pure '\x01d5c7'
    decode "\\<o>" = pure '\x01d5c8'
    decode "\\<p>" = pure '\x01d5c9'
    decode "\\<q>" = pure '\x01d5ca'
    decode "\\<r>" = pure '\x01d5cb'
    decode "\\<s>" = pure '\x01d5cc'
    decode "\\<t>" = pure '\x01d5cd'
    decode "\\<u>" = pure '\x01d5ce'
    decode "\\<v>" = pure '\x01d5cf'
    decode "\\<w>" = pure '\x01d5d0'
    decode "\\<x>" = pure '\x01d5d1'
    decode "\\<y>" = pure '\x01d5d2'
    decode "\\<z>" = pure '\x01d5d3'
    decode "\\<AA>" = pure '\x01d504'
    decode "\\<BB>" = pure '\x01d505'
    decode "\\<CC>" = pure '\x00212d'
    decode "\\<DD>" = pure '\x01d507'
    decode "\\<EE>" = pure '\x01d508'
    decode "\\<FF>" = pure '\x01d509'
    decode "\\<GG>" = pure '\x01d50a'
    decode "\\<HH>" = pure '\x00210c'
    decode "\\<II>" = pure '\x002111'
    decode "\\<JJ>" = pure '\x01d50d'
    decode "\\<KK>" = pure '\x01d50e'
    decode "\\<LL>" = pure '\x01d50f'
    decode "\\<MM>" = pure '\x01d510'
    decode "\\<NN>" = pure '\x01d511'
    decode "\\<OO>" = pure '\x01d512'
    decode "\\<PP>" = pure '\x01d513'
    decode "\\<QQ>" = pure '\x01d514'
    decode "\\<RR>" = pure '\x00211c'
    decode "\\<SS>" = pure '\x01d516'
    decode "\\<TT>" = pure '\x01d517'
    decode "\\<UU>" = pure '\x01d518'
    decode "\\<VV>" = pure '\x01d519'
    decode "\\<WW>" = pure '\x01d51a'
    decode "\\<XX>" = pure '\x01d51b'
    decode "\\<YY>" = pure '\x01d51c'
    decode "\\<ZZ>" = pure '\x002128'
    decode "\\<aa>" = pure '\x01d51e'
    decode "\\<bb>" = pure '\x01d51f'
    decode "\\<cc>" = pure '\x01d520'
    decode "\\<dd>" = pure '\x01d521'
    decode "\\<ee>" = pure '\x01d522'
    decode "\\<ff>" = pure '\x01d523'
    decode "\\<gg>" = pure '\x01d524'
    decode "\\<hh>" = pure '\x01d525'
    decode "\\<ii>" = pure '\x01d526'
    decode "\\<jj>" = pure '\x01d527'
    decode "\\<kk>" = pure '\x01d528'
    decode "\\<ll>" = pure '\x01d529'
    decode "\\<mm>" = pure '\x01d52a'
    decode "\\<nn>" = pure '\x01d52b'
    decode "\\<oo>" = pure '\x01d52c'
    decode "\\<pp>" = pure '\x01d52d'
    decode "\\<qq>" = pure '\x01d52e'
    decode "\\<rr>" = pure '\x01d52f'
    decode "\\<ss>" = pure '\x01d530'
    decode "\\<tt>" = pure '\x01d531'
    decode "\\<uu>" = pure '\x01d532'
    decode "\\<vv>" = pure '\x01d533'
    decode "\\<ww>" = pure '\x01d534'
    decode "\\<xx>" = pure '\x01d535'
    decode "\\<yy>" = pure '\x01d536'
    decode "\\<zz>" = pure '\x01d537'
    decode "\\<alpha>" = pure '\x0003b1'
    decode "\\<beta>" = pure '\x0003b2'
    decode "\\<gamma>" = pure '\x0003b3'
    decode "\\<delta>" = pure '\x0003b4'
    decode "\\<epsilon>" = pure '\x0003b5'
    decode "\\<zeta>" = pure '\x0003b6'
    decode "\\<eta>" = pure '\x0003b7'
    decode "\\<theta>" = pure '\x0003b8'
    decode "\\<iota>" = pure '\x0003b9'
    decode "\\<kappa>" = pure '\x0003ba'
    decode "\\<lambda>" = pure '\x0003bb'
    decode "\\<mu>" = pure '\x0003bc'
    decode "\\<nu>" = pure '\x0003bd'
    decode "\\<xi>" = pure '\x0003be'
    decode "\\<pi>" = pure '\x0003c0'
    decode "\\<rho>" = pure '\x0003c1'
    decode "\\<sigma>" = pure '\x0003c3'
    decode "\\<tau>" = pure '\x0003c4'
    decode "\\<upsilon>" = pure '\x0003c5'
    decode "\\<phi>" = pure '\x0003c6'
    decode "\\<chi>" = pure '\x0003c7'
    decode "\\<psi>" = pure '\x0003c8'
    decode "\\<omega>" = pure '\x0003c9'
    decode "\\<Gamma>" = pure '\x000393'
    decode "\\<Delta>" = pure '\x000394'
    decode "\\<Theta>" = pure '\x000398'
    decode "\\<Lambda>" = pure '\x00039b'
    decode "\\<Xi>" = pure '\x00039e'
    decode "\\<Pi>" = pure '\x0003a0'
    decode "\\<Sigma>" = pure '\x0003a3'
    decode "\\<Upsilon>" = pure '\x0003a5'
    decode "\\<Phi>" = pure '\x0003a6'
    decode "\\<Psi>" = pure '\x0003a8'
    decode "\\<Omega>" = pure '\x0003a9'
    decode "\\<bbbA>" = pure '\x01D538'
    decode "\\<bool>" = pure '\x01d539'
    decode "\\<complex>" = pure '\x002102'
    decode "\\<bbbD>" = pure '\x01D53B'
    decode "\\<bbbE>" = pure '\x01D53C'
    decode "\\<bbbF>" = pure '\x01D53D'
    decode "\\<bbbG>" = pure '\x01D53E'
    decode "\\<bbbH>" = pure '\x00210D'
    decode "\\<bbbI>" = pure '\x01D540'
    decode "\\<bbbJ>" = pure '\x01D541'
    decode "\\<bbbK>" = pure '\x01D542'
    decode "\\<bbbL>" = pure '\x01D543'
    decode "\\<bbbM>" = pure '\x01D544'
    decode "\\<nat>" = pure '\x002115'
    decode "\\<bbbO>" = pure '\x01D546'
    decode "\\<bbbP>" = pure '\x002119'
    decode "\\<rat>" = pure '\x00211a'
    decode "\\<real>" = pure '\x00211d'
    decode "\\<bbbS>" = pure '\x01D54A'
    decode "\\<bbbT>" = pure '\x01D54B'
    decode "\\<bbbU>" = pure '\x01D54C'
    decode "\\<bbbV>" = pure '\x01D54D'
    decode "\\<bbbW>" = pure '\x01D54E'
    decode "\\<bbbX>" = pure '\x01D54F'
    decode "\\<bbbY>" = pure '\x01D550'
    decode "\\<int>" = pure '\x002124'
    decode "\\<leftarrow>" = pure '\x002190'
    decode "\\<longleftarrow>" = pure '\x0027f5'
    decode "\\<longlongleftarrow>" = pure '\x00290e'
    decode "\\<longlonglongleftarrow>" = pure '\x0021e0'
    decode "\\<rightarrow>" = pure '\x002192'
    decode "\\<longrightarrow>" = pure '\x0027f6'
    decode "\\<longlongrightarrow>" = pure '\x00290f'
    decode "\\<longlonglongrightarrow>" = pure '\x0021e2'
    decode "\\<Leftarrow>" = pure '\x0021d0'
    decode "\\<Longleftarrow>" = pure '\x0027f8'
    decode "\\<Lleftarrow>" = pure '\x0021da'
    decode "\\<Rightarrow>" = pure '\x0021d2'
    decode "\\<Longrightarrow>" = pure '\x0027f9'
    decode "\\<Rrightarrow>" = pure '\x0021db'
    decode "\\<leftrightarrow>" = pure '\x002194'
    decode "\\<longleftrightarrow>" = pure '\x0027f7'
    decode "\\<Leftrightarrow>" = pure '\x0021d4'
    decode "\\<Longleftrightarrow>" = pure '\x0027fa'
    decode "\\<mapsto>" = pure '\x0021a6'
    decode "\\<longmapsto>" = pure '\x0027fc'
    decode "\\<midarrow>" = pure '\x002500'
    decode "\\<Midarrow>" = pure '\x002550'
    decode "\\<hookleftarrow>" = pure '\x0021a9'
    decode "\\<hookrightarrow>" = pure '\x0021aa'
    decode "\\<leftharpoondown>" = pure '\x0021bd'
    decode "\\<rightharpoondown>" = pure '\x0021c1'
    decode "\\<leftharpoonup>" = pure '\x0021bc'
    decode "\\<rightharpoonup>" = pure '\x0021c0'
    decode "\\<rightleftharpoons>" = pure '\x0021cc'
    decode "\\<leadsto>" = pure '\x00219d'
    decode "\\<downharpoonleft>" = pure '\x0021c3'
    decode "\\<downharpoonright>" = pure '\x0021c2'
    decode "\\<upharpoonleft>" = pure '\x0021bf'
    decode "\\<upharpoonright>" = pure '\x0021be'
    decode "\\<restriction>" = pure '\x0021be'
    decode "\\<Colon>" = pure '\x002237'
    decode "\\<up>" = pure '\x002191'
    decode "\\<Up>" = pure '\x0021d1'
    decode "\\<down>" = pure '\x002193'
    decode "\\<Down>" = pure '\x0021d3'
    decode "\\<updown>" = pure '\x002195'
    decode "\\<Updown>" = pure '\x0021d5'
    decode "\\<langle>" = pure '\x0027e8'
    decode "\\<rangle>" = pure '\x0027e9'
    decode "\\<llangle>" = pure '\x0027ea'
    decode "\\<rrangle>" = pure '\x0027eb'
    decode "\\<lceil>" = pure '\x002308'
    decode "\\<rceil>" = pure '\x002309'
    decode "\\<lfloor>" = pure '\x00230a'
    decode "\\<rfloor>" = pure '\x00230b'
    decode "\\<lparr>" = pure '\x002987'
    decode "\\<rparr>" = pure '\x002988'
    decode "\\<lbrakk>" = pure '\x0027e6'
    decode "\\<rbrakk>" = pure '\x0027e7'
    decode "\\<lbrace>" = pure '\x002983'
    decode "\\<rbrace>" = pure '\x002984'
    decode "\\<lblot>" = pure '\x002989'
    decode "\\<rblot>" = pure '\x00298A'
    decode "\\<guillemotleft>" = pure '\x0000ab'
    decode "\\<guillemotright>" = pure '\x0000bb'
    decode "\\<bottom>" = pure '\x0022a5'
    decode "\\<top>" = pure '\x0022a4'
    decode "\\<and>" = pure '\x002227'
    decode "\\<And>" = pure '\x0022c0'
    decode "\\<or>" = pure '\x002228'
    decode "\\<Or>" = pure '\x0022c1'
    decode "\\<forall>" = pure '\x002200'
    decode "\\<exists>" = pure '\x002203'
    decode "\\<nexists>" = pure '\x002204'
    decode "\\<not>" = pure '\x0000ac'
    decode "\\<circle>" = pure '\x0025cb'
    decode "\\<box>" = pure '\x0025a1'
    decode "\\<diamond>" = pure '\x0025c7'
    decode "\\<diamondop>" = pure '\x0022c4'
    decode "\\<turnstile>" = pure '\x0022a2'
    decode "\\<Turnstile>" = pure '\x0022a8'
    decode "\\<tturnstile>" = pure '\x0022a9'
    decode "\\<TTurnstile>" = pure '\x0022ab'
    decode "\\<stileturn>" = pure '\x0022a3'
    decode "\\<surd>" = pure '\x00221a'
    decode "\\<le>" = pure '\x002264'
    decode "\\<ge>" = pure '\x002265'
    decode "\\<lless>" = pure '\x00226a'
    decode "\\<ggreater>" = pure '\x00226b'
    decode "\\<lesssim>" = pure '\x002272'
    decode "\\<greatersim>" = pure '\x002273'
    decode "\\<lessapprox>" = pure '\x002a85'
    decode "\\<greaterapprox>" = pure '\x002a86'
    decode "\\<in>" = pure '\x002208'
    decode "\\<notin>" = pure '\x002209'
    decode "\\<subset>" = pure '\x002282'
    decode "\\<supset>" = pure '\x002283'
    decode "\\<subseteq>" = pure '\x002286'
    decode "\\<supseteq>" = pure '\x002287'
    decode "\\<sqsubset>" = pure '\x00228f'
    decode "\\<sqsupset>" = pure '\x002290'
    decode "\\<sqsubseteq>" = pure '\x002291'
    decode "\\<sqsupseteq>" = pure '\x002292'
    decode "\\<inter>" = pure '\x002229'
    decode "\\<Inter>" = pure '\x0022c2'
    decode "\\<union>" = pure '\x00222a'
    decode "\\<Union>" = pure '\x0022c3'
    decode "\\<squnion>" = pure '\x002294'
    decode "\\<Squnion>" = pure '\x002a06'
    decode "\\<sqinter>" = pure '\x002293'
    decode "\\<Sqinter>" = pure '\x002a05'
    decode "\\<setminus>" = pure '\x002216'
    decode "\\<propto>" = pure '\x00221d'
    decode "\\<uplus>" = pure '\x00228e'
    decode "\\<Uplus>" = pure '\x002a04'
    decode "\\<noteq>" = pure '\x002260'
    decode "\\<sim>" = pure '\x00223c'
    decode "\\<doteq>" = pure '\x002250'
    decode "\\<simeq>" = pure '\x002243'
    decode "\\<approx>" = pure '\x002248'
    decode "\\<asymp>" = pure '\x00224d'
    decode "\\<cong>" = pure '\x002245'
    decode "\\<smile>" = pure '\x002323'
    decode "\\<equiv>" = pure '\x002261'
    decode "\\<frown>" = pure '\x002322'
    decode "\\<Join>" = pure '\x0022c8'
    decode "\\<bowtie>" = pure '\x002a1d'
    decode "\\<prec>" = pure '\x00227a'
    decode "\\<succ>" = pure '\x00227b'
    decode "\\<preceq>" = pure '\x00227c'
    decode "\\<succeq>" = pure '\x00227d'
    decode "\\<parallel>" = pure '\x002225'
    decode "\\<Parallel>" = pure '\x002016'
    decode "\\<interleave>" = pure '\x002af4'
    decode "\\<sslash>" = pure '\x002afd'
    decode "\\<bar>" = pure '\x0000a6'
    decode "\\<bbar>" = pure '\x002aff'
    decode "\\<plusminus>" = pure '\x0000b1'
    decode "\\<minusplus>" = pure '\x002213'
    decode "\\<times>" = pure '\x0000d7'
    decode "\\<div>" = pure '\x0000f7'
    decode "\\<cdot>" = pure '\x0022c5'
    decode "\\<star>" = pure '\x0022c6'
    decode "\\<bullet>" = pure '\x002219'
    decode "\\<circ>" = pure '\x002218'
    decode "\\<dagger>" = pure '\x002020'
    decode "\\<ddagger>" = pure '\x002021'
    decode "\\<lhd>" = pure '\x0022b2'
    decode "\\<rhd>" = pure '\x0022b3'
    decode "\\<unlhd>" = pure '\x0022b4'
    decode "\\<unrhd>" = pure '\x0022b5'
    decode "\\<triangleleft>" = pure '\x0025c3'
    decode "\\<triangleright>" = pure '\x0025b9'
    decode "\\<triangle>" = pure '\x0025b3'
    decode "\\<triangleq>" = pure '\x00225c'
    decode "\\<oplus>" = pure '\x002295'
    decode "\\<Oplus>" = pure '\x002a01'
    decode "\\<otimes>" = pure '\x002297'
    decode "\\<Otimes>" = pure '\x002a02'
    decode "\\<odot>" = pure '\x002299'
    decode "\\<Odot>" = pure '\x002a00'
    decode "\\<ominus>" = pure '\x002296'
    decode "\\<oslash>" = pure '\x002298'
    decode "\\<dots>" = pure '\x002026'
    decode "\\<cdots>" = pure '\x0022ef'
    decode "\\<Sum>" = pure '\x002211'
    decode "\\<Prod>" = pure '\x00220f'
    decode "\\<Coprod>" = pure '\x002210'
    decode "\\<infinity>" = pure '\x00221e'
    decode "\\<integral>" = pure '\x00222b'
    decode "\\<ointegral>" = pure '\x00222e'
    decode "\\<clubsuit>" = pure '\x002663'
    decode "\\<diamondsuit>" = pure '\x002662'
    decode "\\<heartsuit>" = pure '\x002661'
    decode "\\<spadesuit>" = pure '\x002660'
    decode "\\<aleph>" = pure '\x002135'
    decode "\\<emptyset>" = pure '\x002205'
    decode "\\<nabla>" = pure '\x002207'
    decode "\\<partial>" = pure '\x002202'
    decode "\\<flat>" = pure '\x00266d'
    decode "\\<natural>" = pure '\x00266e'
    decode "\\<sharp>" = pure '\x00266f'
    decode "\\<angle>" = pure '\x002220'
    decode "\\<copyright>" = pure '\x0000a9'
    decode "\\<registered>" = pure '\x0000ae'
    decode "\\<hyphen>" = pure '\x002010'
    decode "\\<inverse>" = pure '\x0000af'
    decode "\\<sqdot>" = pure '\x0000b7'
    decode "\\<onequarter>" = pure '\x0000bc'
    decode "\\<onehalf>" = pure '\x0000bd'
    decode "\\<threequarters>" = pure '\x0000be'
    decode "\\<ordfeminine>" = pure '\x0000aa'
    decode "\\<ordmasculine>" = pure '\x0000ba'
    decode "\\<section>" = pure '\x0000a7'
    decode "\\<paragraph>" = pure '\x0000b6'
    decode "\\<exclamdown>" = pure '\x0000a1'
    decode "\\<questiondown>" = pure '\x0000bf'
    decode "\\<euro>" = pure '\x0020ac'
    decode "\\<pounds>" = pure '\x0000a3'
    decode "\\<yen>" = pure '\x0000a5'
    decode "\\<cent>" = pure '\x0000a2'
    decode "\\<currency>" = pure '\x0000a4'
    decode "\\<degree>" = pure '\x0000b0'
    decode "\\<amalg>" = pure '\x002a3f'
    decode "\\<mho>" = pure '\x002127'
    decode "\\<lozenge>" = pure '\x0025ca'
    decode "\\<wp>" = pure '\x002118'
    decode "\\<wrong>" = pure '\x002240'
    decode "\\<acute>" = pure '\x0000b4'
    decode "\\<index>" = pure '\x000131'
    decode "\\<dieresis>" = pure '\x0000a8'
    decode "\\<cedilla>" = pure '\x0000b8'
    decode "\\<hungarumlaut>" = pure '\x0002dd'
    decode "\\<bind>" = pure '\x00291c'
    decode "\\<then>" = pure '\x002aa2'
    decode "\\<some>" = pure '\x0003f5'
    decode "\\<Zcomp>" = pure '\x002A3E'
    decode "\\<Zinj>" = pure '\x0021A3'
    decode "\\<Zpinj>" = pure '\x002914'
    decode "\\<Zfinj>" = pure '\x002915'
    decode "\\<Zsurj>" = pure '\x0021A0'
    decode "\\<Zpsurj>" = pure '\x002900'
    decode "\\<Zbij>" = pure '\x002916'
    decode "\\<Zpfun>" = pure '\x0021F8'
    decode "\\<Zffun>" = pure '\x0021FB'
    decode "\\<Zdres>" = pure '\x0025C1'
    decode "\\<Zndres>" = pure '\x002A64'
    decode "\\<Zrres>" = pure '\x0025B7'
    decode "\\<Znrres>" = pure '\x002A65'
    decode "\\<Zspot>" = pure '\x002981'
    decode "\\<Zsemi>" = pure '\x002A1F'
    decode "\\<Zproject>" = pure '\x002A21'
    decode "\\<Ztypecolon>" = pure '\x002982'
    decode "\\<Zhide>" = pure '\x0029F9'
    decode "\\<Zcat>" = pure '\x002040'
    decode "\\<Zinbag>" = pure '\x0022FF'
    decode "\\<Zprime>" = pure '\x002032'
    decode "\\<hole>" = pure '\x002311'
    decode "\\<newline>" = pure '\x0023ce'
    decode "\\<comment>" = pure '\x002015'
    decode "\\<^cancel>" = pure '\x002326'
    decode "\\<^marker>" = pure '\x002710'
    decode "\\<checkmark>" = pure '\x002713'
    decode "\\<crossmark>" = pure '\x002717'
    decode "\\<open>" = pure '\x002039'
    decode "\\<close>" = pure '\x00203a'
    decode "\\<^here>" = pure '\x002302'
    decode "\\<^undefined>" = pure '\x002756'
    decode "\\<^noindent>" = pure '\x0021e4'
    decode "\\<^smallskip>" = pure '\x002508'
    decode "\\<^medskip>" = pure '\x002509'
    decode "\\<^bigskip>" = pure '\x002501'
    decode "\\<^item>" = pure '\x0025aa'
    decode "\\<^enum>" = pure '\x0025b8'
    decode "\\<^descr>" = pure '\x0027a7'
    decode "\\<^footnote>" = pure '\x00204b'
    decode "\\<^verbatim>" = pure '\x0025a9'
    decode "\\<^theory_text>" = pure '\x002b1a'
    decode "\\<^emph>" = pure '\x002217'
    decode "\\<^bold>" = pure '\x002759'
    decode "\\<^sub>" = pure '\x0021e9'
    decode "\\<^sup>" = pure '\x0021e7'
    decode "\\<^bsub>" = pure '\x0021d8'
    decode "\\<^esub>" = pure '\x0021d9'
    decode "\\<^bsup>" = pure '\x0021d7'
    decode "\\<^esup>" = pure '\x0021d6'
    decode "\\<^file>" = pure '\x01F5CF'
    decode "\\<^dir>" = pure '\x01F5C0'
    decode "\\<^url>" = pure '\x01F310'
    decode "\\<^doc>" = pure '\x01F4D3'
    decode "\\<^action>" = pure '\x00261b'
    decode [c] = pure c
    decode str = Message.errorLexer noPos $
      "The input text contains an invalid character sequence \"" <> str <> "\"."
